// This project is released under the GPL Licence

/*****************************************************************************/

#include <stdlib.h>
#include <string.h>
#include <math.h>

/*****************************************************************************/

#include "ladspa.h"

/*****************************************************************************/

/* The port numbers for the plugin: */

#define PXU_CONTROL1 0
#define PXU_CONTROL2 1
#define PXU_CONTROL3 2
#define PXU_CONTROL4 3
#define PXU_CONTROL5 4
#define PXU_CONTROL6 5
#define PXU_CONTROL7 6
#define PXU_CONTROL8 7
#define PXU_CONTROL9 8
#define PXU_CONTROL10 9

#define PXU_INPUT1  10
#define PXU_OUTPUT1 11
#define PXU_INPUT2  12
#define PXU_OUTPUT2 13

/*****************************************************************************/
	
	
/* The structure used to hold port connection information and state
   (actually gain controls require no further state). */

typedef struct {

  /* Ports:
     ------ */
  LADSPA_Data * m_pfControlValue1;
  LADSPA_Data * m_pfControlValue2;
  LADSPA_Data * m_pfControlValue3;
  LADSPA_Data * m_pfControlValue4;
  LADSPA_Data * m_pfControlValue5;
  LADSPA_Data * m_pfControlValue6;
  LADSPA_Data * m_pfControlValue7;
  LADSPA_Data * m_pfControlValue8;
  LADSPA_Data * m_pfControlValue9;
  LADSPA_Data * m_pfControlValue10;
  LADSPA_Data * m_pfInputBuffer1;
  LADSPA_Data * m_pfOutputBuffer1;
  LADSPA_Data * m_pfInputBuffer2;  /* (Not used for mono) */
  LADSPA_Data * m_pfOutputBuffer2; /* (Not used for mono) */

  
} Pxu;


/*****************************************************************************/


LADSPA_Handle 
instantiatePxu(const LADSPA_Descriptor * Descriptor,
			   unsigned long             SampleRate) {

  Pxu * psPxu;

  psPxu 
    = (Pxu *)malloc(sizeof(Pxu));

  if (psPxu == NULL) 
    return NULL;
  
  return psPxu;
}


/*****************************************************************************/

/* Connect a port to a data location. */
void 
connectPortToPxu(LADSPA_Handle Instance,
		       unsigned long Port,
		       LADSPA_Data * DataLocation) {

  Pxu * psPxu;

  psPxu = (Pxu *)Instance;
  switch (Port) {
  case PXU_CONTROL1:
    psPxu->m_pfControlValue1 = DataLocation;  
    break;
  case PXU_CONTROL2:
    psPxu->m_pfControlValue2 = DataLocation; 
    break;
  case PXU_CONTROL3:
    psPxu->m_pfControlValue3 = DataLocation; 
    break;
  case PXU_CONTROL4:
    psPxu->m_pfControlValue4 = DataLocation; 
    break;
  case PXU_CONTROL5:
    psPxu->m_pfControlValue5 = DataLocation; 
    break;
  case PXU_CONTROL6:
    psPxu->m_pfControlValue6 = DataLocation; 
    break;
  case PXU_CONTROL7:
    psPxu->m_pfControlValue7 = DataLocation; 
    break;
  case PXU_CONTROL8:
    psPxu->m_pfControlValue8 = DataLocation; 
    break;
  case PXU_CONTROL9:
    psPxu->m_pfControlValue9 = DataLocation; 
    break;
  case PXU_CONTROL10:
    psPxu->m_pfControlValue10 = DataLocation; 
    break;
  case PXU_INPUT1:
    psPxu->m_pfInputBuffer1 = DataLocation;
    break;
  case PXU_OUTPUT1:
    psPxu->m_pfOutputBuffer1 = DataLocation;
    break;
  case PXU_INPUT2:
    /* (This should only happen for stereo.) */
    psPxu->m_pfInputBuffer2 = DataLocation;
    break;
  case PXU_OUTPUT2:
    /* (This should only happen for stereo.) */
    psPxu->m_pfOutputBuffer2 = DataLocation;
    break;
  }

}

/*****************************************************************************/

void 
runMonoPxu(LADSPA_Handle Instance,
		 unsigned long SampleCount) {
  
  LADSPA_Data * pfInput;
  LADSPA_Data * pfOutput;
  LADSPA_Data fVar1;
  LADSPA_Data fVar2;
  LADSPA_Data fVar3;
  LADSPA_Data fVar4;
  LADSPA_Data fVar5;
  LADSPA_Data fVar6;
  LADSPA_Data fVar7;
  LADSPA_Data fVar8;
  LADSPA_Data fVar9;
  LADSPA_Data fVar10;
  Pxu * psPxu;
  unsigned long lSampleIndex;

  psPxu = (Pxu *)Instance;

  pfInput = psPxu->m_pfInputBuffer1;
  pfOutput = psPxu->m_pfOutputBuffer1;
  fVar1 = *(psPxu->m_pfControlValue1);
  fVar2 = *(psPxu->m_pfControlValue2);
  fVar3 = *(psPxu->m_pfControlValue3);
  fVar4 = *(psPxu->m_pfControlValue4);
  fVar5 = *(psPxu->m_pfControlValue5);
  fVar6 = *(psPxu->m_pfControlValue6);
  fVar7 = *(psPxu->m_pfControlValue7);
  fVar8 = *(psPxu->m_pfControlValue8);
  fVar9 = *(psPxu->m_pfControlValue9);
  fVar10 = *(psPxu->m_pfControlValue10);

  for (lSampleIndex = 0; lSampleIndex < SampleCount; lSampleIndex++) 
    *(pfOutput++) = *(pfInput++) * fVar1;
}

/*****************************************************************************/

void 
runStereoPxu(LADSPA_Handle Instance,
		   unsigned long SampleCount) {
  
  LADSPA_Data * pfInput1;
  LADSPA_Data * pfInput2;
  LADSPA_Data * pfOutput1;
  LADSPA_Data * pfOutput2;
  LADSPA_Data fVar1;
  LADSPA_Data fVar2;
  LADSPA_Data fVar3;
  LADSPA_Data fVar4;
  LADSPA_Data fVar5;
  LADSPA_Data fVar6;
  LADSPA_Data fVar7;
  LADSPA_Data fVar8;
  LADSPA_Data fVar9;
  LADSPA_Data fVar10;
  
  Pxu * psPxu;
  unsigned long lSampleIndex;

  psPxu = (Pxu *)Instance;
  
  fVar1 = *(psPxu->m_pfControlValue1);
  fVar2 = *(psPxu->m_pfControlValue2);
  fVar3 = *(psPxu->m_pfControlValue3);
  fVar4 = *(psPxu->m_pfControlValue4);
  fVar5 = *(psPxu->m_pfControlValue5);
  fVar6 = *(psPxu->m_pfControlValue6);
  fVar7 = *(psPxu->m_pfControlValue7);
  fVar8 = *(psPxu->m_pfControlValue8);
  fVar9 = *(psPxu->m_pfControlValue9);
  fVar10 = *(psPxu->m_pfControlValue10);
  
  pfInput1 = psPxu->m_pfInputBuffer1;
  pfInput2 = psPxu->m_pfInputBuffer2;
  pfOutput1 = psPxu->m_pfOutputBuffer1;
  pfOutput2 = psPxu->m_pfOutputBuffer2;
  
  double b_thr = pow(2,(((fVar1-0.5)*100))/ 6); 
  double b_kn = pow(2,(((fVar2)*-100))/ 6); 
  
  for (lSampleIndex = 0; lSampleIndex < SampleCount; lSampleIndex++) 
 {
   double b_in1 = *(pfInput1++); 
   double b_in2 = *(pfInput2++); 

   b_in1 = b_in1 * b_thr;
   if (b_in1 > 1) {b_in1 = 1;} else if (b_in1 < -1) {b_in1 = -1;}
   double b_inr = b_in1;
   if (b_inr < 0) {b_inr = -b_inr;}
   if (b_inr > b_kn) {
     b_inr = b_inr - b_kn;
     b_inr = b_inr / (1-b_kn);
     b_inr = 1-b_inr;
     b_inr = b_inr*b_inr;
     b_inr = 1-b_inr;
     b_inr = b_inr * 0.5;
     b_inr = b_inr * (1-b_kn);
     b_inr = b_inr + b_kn;
     if (b_in1 < 0) {b_inr = -b_inr;}
     b_in1 = b_inr;
   }
   b_in1 = b_in1 / b_thr;

   b_in2 = b_in2 * b_thr;
   if (b_in2 > 1) {b_in2 = 1;} else if (b_in2 < -1) {b_in2 = -1;}
   b_inr = b_in2;
   if (b_inr < 0) {b_inr = -b_inr;}
   if (b_inr > b_kn) {
     b_inr = b_inr - b_kn;
     b_inr = b_inr / (1-b_kn);
     b_inr = 1-b_inr;
     b_inr = b_inr*b_inr;
     b_inr = 1-b_inr;
     b_inr = b_inr * 0.5;
     b_inr = b_inr * (1-b_kn);
     b_inr = b_inr + b_kn;
     if (b_in2 < 0) {b_inr = -b_inr;}
     b_in2 = b_inr;
   }
   b_in2 = b_in2 / b_thr;
   
   *(pfOutput1++) = b_in1;
   *(pfOutput2++) = b_in2;
 }


}

/*****************************************************************************/

/* Throw away a simple delay line. */
void 
cleanupPxu(LADSPA_Handle Instance) {
  free(Instance);
}

/*****************************************************************************/

LADSPA_Descriptor * g_psMonoDescriptor = NULL;
LADSPA_Descriptor * g_psStereoDescriptor = NULL;

/*****************************************************************************/

/* _init() is called automatically when the plugin library is first
   loaded. */
void 
_init() {

  char ** pcPortNames;
  LADSPA_PortDescriptor * piPortDescriptors;
  LADSPA_PortRangeHint * psPortRangeHints;

  g_psMonoDescriptor
    = (LADSPA_Descriptor *)malloc(sizeof(LADSPA_Descriptor));
  g_psStereoDescriptor 
    = (LADSPA_Descriptor *)malloc(sizeof(LADSPA_Descriptor));

  if (g_psMonoDescriptor) {
  
    g_psMonoDescriptor->UniqueID
      = 994;                                         	
    g_psMonoDescriptor->Label
      = strdup("Mlm_Saturator_M");				
    g_psMonoDescriptor->Properties
      = LADSPA_PROPERTY_HARD_RT_CAPABLE;
    g_psMonoDescriptor->Name 
      = strdup("Millennium Saturator [mono] (not implemented)");			
    g_psMonoDescriptor->Maker
      = strdup("Opensource GPL");
    g_psMonoDescriptor->Copyright
      = strdup("Opensource GPL");
    g_psMonoDescriptor->PortCount
      = 12;						
    piPortDescriptors
      = (LADSPA_PortDescriptor *)calloc(12, sizeof(LADSPA_PortDescriptor));
    g_psMonoDescriptor->PortDescriptors
      = (const LADSPA_PortDescriptor *)piPortDescriptors;
      
    piPortDescriptors[PXU_CONTROL1]
      = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    piPortDescriptors[PXU_CONTROL2]
      = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    piPortDescriptors[PXU_CONTROL3]
      = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    piPortDescriptors[PXU_CONTROL4]
      = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    piPortDescriptors[PXU_CONTROL5]
      = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    piPortDescriptors[PXU_CONTROL6]
      = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    piPortDescriptors[PXU_CONTROL7]
      = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    piPortDescriptors[PXU_CONTROL8]
      = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    piPortDescriptors[PXU_CONTROL9]
      = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    piPortDescriptors[PXU_CONTROL10]
      = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
      
    piPortDescriptors[PXU_INPUT1]
      = LADSPA_PORT_INPUT | LADSPA_PORT_AUDIO;
    piPortDescriptors[PXU_OUTPUT1]
      = LADSPA_PORT_OUTPUT | LADSPA_PORT_AUDIO;
    pcPortNames
      = (char **)calloc(12, sizeof(char *));
    g_psMonoDescriptor->PortNames 
      = (const char **)pcPortNames;
    pcPortNames[PXU_CONTROL1]
      = strdup("1");
    pcPortNames[PXU_CONTROL2]
      = strdup("2");
    pcPortNames[PXU_CONTROL3]
      = strdup("3");
    pcPortNames[PXU_CONTROL4]
      = strdup("4");
    pcPortNames[PXU_CONTROL5]
      = strdup("5");
    pcPortNames[PXU_CONTROL6]
      = strdup("6");
    pcPortNames[PXU_CONTROL7]
      = strdup("7");
    pcPortNames[PXU_CONTROL8]
      = strdup("8");
    pcPortNames[PXU_CONTROL9]
      = strdup("9");
    pcPortNames[PXU_CONTROL10]
      = strdup("10");
    pcPortNames[PXU_INPUT1]
      = strdup("Input");
    pcPortNames[PXU_OUTPUT1]
      = strdup("Output");
    psPortRangeHints = ((LADSPA_PortRangeHint *)
			calloc(12, sizeof(LADSPA_PortRangeHint)));
    g_psMonoDescriptor->PortRangeHints
      = (const LADSPA_PortRangeHint *)psPortRangeHints;
      
    psPortRangeHints[PXU_CONTROL1].HintDescriptor
      = (LADSPA_HINT_BOUNDED_BELOW 
	 | LADSPA_HINT_BOUNDED_ABOVE
	 | LADSPA_HINT_DEFAULT_MIDDLE);
    psPortRangeHints[PXU_CONTROL1].LowerBound 
      = 0;
    psPortRangeHints[PXU_CONTROL1].UpperBound
      = 1;

    psPortRangeHints[PXU_CONTROL2].HintDescriptor
      = (LADSPA_HINT_BOUNDED_BELOW 
	 | LADSPA_HINT_BOUNDED_ABOVE
	 | LADSPA_HINT_DEFAULT_MIDDLE);
    psPortRangeHints[PXU_CONTROL2].LowerBound 
      = 0;
    psPortRangeHints[PXU_CONTROL2].UpperBound
      = 1;

    psPortRangeHints[PXU_CONTROL3].HintDescriptor
      = (LADSPA_HINT_BOUNDED_BELOW 
	 | LADSPA_HINT_BOUNDED_ABOVE
	 | LADSPA_HINT_DEFAULT_MIDDLE);
    psPortRangeHints[PXU_CONTROL3].LowerBound 
      = 0;
    psPortRangeHints[PXU_CONTROL3].UpperBound
      = 1;

    psPortRangeHints[PXU_CONTROL4].HintDescriptor
      = (LADSPA_HINT_BOUNDED_BELOW 
	 | LADSPA_HINT_BOUNDED_ABOVE
	 | LADSPA_HINT_DEFAULT_MIDDLE);
    psPortRangeHints[PXU_CONTROL4].LowerBound 
      = 0;
    psPortRangeHints[PXU_CONTROL4].UpperBound
      = 1;

    psPortRangeHints[PXU_CONTROL5].HintDescriptor
      = (LADSPA_HINT_BOUNDED_BELOW 
	 | LADSPA_HINT_BOUNDED_ABOVE
	 | LADSPA_HINT_DEFAULT_MIDDLE);
    psPortRangeHints[PXU_CONTROL5].LowerBound 
      = 0;
    psPortRangeHints[PXU_CONTROL5].UpperBound
      = 1;

    psPortRangeHints[PXU_CONTROL6].HintDescriptor
      = (LADSPA_HINT_BOUNDED_BELOW 
	 | LADSPA_HINT_BOUNDED_ABOVE
	 | LADSPA_HINT_DEFAULT_MIDDLE);
    psPortRangeHints[PXU_CONTROL6].LowerBound 
      = 0;
    psPortRangeHints[PXU_CONTROL6].UpperBound
      = 1;

    psPortRangeHints[PXU_CONTROL7].HintDescriptor
      = (LADSPA_HINT_BOUNDED_BELOW 
	 | LADSPA_HINT_BOUNDED_ABOVE
	 | LADSPA_HINT_DEFAULT_MIDDLE);
    psPortRangeHints[PXU_CONTROL7].LowerBound 
      = 0;
    psPortRangeHints[PXU_CONTROL7].UpperBound
      = 1;

    psPortRangeHints[PXU_CONTROL8].HintDescriptor
      = (LADSPA_HINT_BOUNDED_BELOW 
	 | LADSPA_HINT_BOUNDED_ABOVE
	 | LADSPA_HINT_DEFAULT_MIDDLE);
    psPortRangeHints[PXU_CONTROL8].LowerBound 
      = 0;
    psPortRangeHints[PXU_CONTROL8].UpperBound
      = 1;

    psPortRangeHints[PXU_CONTROL9].HintDescriptor
      = (LADSPA_HINT_BOUNDED_BELOW 
	 | LADSPA_HINT_BOUNDED_ABOVE
	 | LADSPA_HINT_DEFAULT_MIDDLE);
    psPortRangeHints[PXU_CONTROL9].LowerBound 
      = 0;
    psPortRangeHints[PXU_CONTROL9].UpperBound
      = 1;

    psPortRangeHints[PXU_CONTROL10].HintDescriptor
      = (LADSPA_HINT_BOUNDED_BELOW 
	 | LADSPA_HINT_BOUNDED_ABOVE
	 | LADSPA_HINT_DEFAULT_MIDDLE);
    psPortRangeHints[PXU_CONTROL10].LowerBound 
      = 0;
    psPortRangeHints[PXU_CONTROL10].UpperBound
      = 1;
      
      
    psPortRangeHints[PXU_INPUT1].HintDescriptor
      = 0;
    psPortRangeHints[PXU_OUTPUT1].HintDescriptor
      = 0;
    g_psMonoDescriptor->instantiate 
      = instantiatePxu;
    g_psMonoDescriptor->connect_port 
      = connectPortToPxu;
    g_psMonoDescriptor->activate
      = NULL;
    g_psMonoDescriptor->run
      = runMonoPxu;
    g_psMonoDescriptor->run_adding
      = NULL;
    g_psMonoDescriptor->set_run_adding_gain
      = NULL;
    g_psMonoDescriptor->deactivate
      = NULL;
    g_psMonoDescriptor->cleanup
      = cleanupPxu;
  }
  
  if (g_psStereoDescriptor) {
    
    g_psStereoDescriptor->UniqueID
      = 995;
    g_psStereoDescriptor->Label
      = strdup("Mlm_Saturator_S");
    g_psStereoDescriptor->Properties
      = LADSPA_PROPERTY_HARD_RT_CAPABLE;
    g_psStereoDescriptor->Name 
      = strdup("Millennium Saturator [stereo]");
    g_psStereoDescriptor->Maker
      = strdup("Opensource GPL)");
    g_psStereoDescriptor->Copyright
      = strdup("Opensource GPL");
    g_psStereoDescriptor->PortCount
      = 14;
    piPortDescriptors
      = (LADSPA_PortDescriptor *)calloc(14, sizeof(LADSPA_PortDescriptor));
    g_psStereoDescriptor->PortDescriptors
      = (const LADSPA_PortDescriptor *)piPortDescriptors;
    piPortDescriptors[PXU_CONTROL1]
      = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    piPortDescriptors[PXU_CONTROL2]
      = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    piPortDescriptors[PXU_CONTROL3]
      = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    piPortDescriptors[PXU_CONTROL4]
      = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    piPortDescriptors[PXU_CONTROL5]
      = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    piPortDescriptors[PXU_CONTROL6]
      = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    piPortDescriptors[PXU_CONTROL7]
      = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    piPortDescriptors[PXU_CONTROL8]
      = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    piPortDescriptors[PXU_CONTROL9]
      = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    piPortDescriptors[PXU_CONTROL10]
      = LADSPA_PORT_INPUT | LADSPA_PORT_CONTROL;
    piPortDescriptors[PXU_INPUT1]
      = LADSPA_PORT_INPUT | LADSPA_PORT_AUDIO;
    piPortDescriptors[PXU_OUTPUT1]
      = LADSPA_PORT_OUTPUT | LADSPA_PORT_AUDIO;
    piPortDescriptors[PXU_INPUT2]
      = LADSPA_PORT_INPUT | LADSPA_PORT_AUDIO;
    piPortDescriptors[PXU_OUTPUT2]
      = LADSPA_PORT_OUTPUT | LADSPA_PORT_AUDIO;
    pcPortNames
      = (char **)calloc(14, sizeof(char *));
    g_psStereoDescriptor->PortNames 
      = (const char **)pcPortNames;
    pcPortNames[PXU_CONTROL1]
      = strdup("Threshold");
    pcPortNames[PXU_CONTROL2]
      = strdup("Knee");
    pcPortNames[PXU_CONTROL3]
      = strdup("3");
    pcPortNames[PXU_CONTROL4]
      = strdup("4");
    pcPortNames[PXU_CONTROL5]
      = strdup("5");
    pcPortNames[PXU_CONTROL6]
      = strdup("6");
    pcPortNames[PXU_CONTROL7]
      = strdup("7");
    pcPortNames[PXU_CONTROL8]
      = strdup("8");
    pcPortNames[PXU_CONTROL9]
      = strdup("9");
    pcPortNames[PXU_CONTROL10]
      = strdup("10");
    pcPortNames[PXU_INPUT1]
      = strdup("Input (Left)");
    pcPortNames[PXU_OUTPUT1]
      = strdup("Output (Left)");
    pcPortNames[PXU_INPUT2]
      = strdup("Input (Right)");
    pcPortNames[PXU_OUTPUT2]
      = strdup("Output (Right)");
    psPortRangeHints = ((LADSPA_PortRangeHint *)
			calloc(14, sizeof(LADSPA_PortRangeHint)));
    g_psStereoDescriptor->PortRangeHints
      = (const LADSPA_PortRangeHint *)psPortRangeHints;

    psPortRangeHints[PXU_CONTROL1].HintDescriptor
      = (LADSPA_HINT_BOUNDED_BELOW 
	 | LADSPA_HINT_BOUNDED_ABOVE
	 | LADSPA_HINT_DEFAULT_MIDDLE);
    psPortRangeHints[PXU_CONTROL1].LowerBound 
      = 0;
    psPortRangeHints[PXU_CONTROL1].UpperBound
      = 1;

    psPortRangeHints[PXU_CONTROL2].HintDescriptor
      = (LADSPA_HINT_BOUNDED_BELOW 
	 | LADSPA_HINT_BOUNDED_ABOVE
	 | LADSPA_HINT_DEFAULT_MIDDLE);
    psPortRangeHints[PXU_CONTROL2].LowerBound 
      = 0;
    psPortRangeHints[PXU_CONTROL2].UpperBound
      = 1;

    psPortRangeHints[PXU_CONTROL3].HintDescriptor
      = (LADSPA_HINT_BOUNDED_BELOW 
	 | LADSPA_HINT_BOUNDED_ABOVE
	 | LADSPA_HINT_DEFAULT_MIDDLE);
    psPortRangeHints[PXU_CONTROL3].LowerBound 
      = 0;
    psPortRangeHints[PXU_CONTROL3].UpperBound
      = 1;

    psPortRangeHints[PXU_CONTROL4].HintDescriptor
      = (LADSPA_HINT_BOUNDED_BELOW 
	 | LADSPA_HINT_BOUNDED_ABOVE
	 | LADSPA_HINT_DEFAULT_MIDDLE);
    psPortRangeHints[PXU_CONTROL4].LowerBound 
      = 0;
    psPortRangeHints[PXU_CONTROL4].UpperBound
      = 1;

    psPortRangeHints[PXU_CONTROL5].HintDescriptor
      = (LADSPA_HINT_BOUNDED_BELOW 
	 | LADSPA_HINT_BOUNDED_ABOVE
	 | LADSPA_HINT_DEFAULT_MIDDLE);
    psPortRangeHints[PXU_CONTROL5].LowerBound 
      = 0;
    psPortRangeHints[PXU_CONTROL5].UpperBound
      = 1;

    psPortRangeHints[PXU_CONTROL6].HintDescriptor
      = (LADSPA_HINT_BOUNDED_BELOW 
	 | LADSPA_HINT_BOUNDED_ABOVE
	 | LADSPA_HINT_DEFAULT_MIDDLE);
    psPortRangeHints[PXU_CONTROL6].LowerBound 
      = 0;
    psPortRangeHints[PXU_CONTROL6].UpperBound
      = 1;

    psPortRangeHints[PXU_CONTROL7].HintDescriptor
      = (LADSPA_HINT_BOUNDED_BELOW 
	 | LADSPA_HINT_BOUNDED_ABOVE
	 | LADSPA_HINT_DEFAULT_MIDDLE);
    psPortRangeHints[PXU_CONTROL7].LowerBound 
      = 0;
    psPortRangeHints[PXU_CONTROL7].UpperBound
      = 1;

    psPortRangeHints[PXU_CONTROL8].HintDescriptor
      = (LADSPA_HINT_BOUNDED_BELOW 
	 | LADSPA_HINT_BOUNDED_ABOVE
	 | LADSPA_HINT_DEFAULT_MIDDLE);
    psPortRangeHints[PXU_CONTROL8].LowerBound 
      = 0;
    psPortRangeHints[PXU_CONTROL8].UpperBound
      = 1;

    psPortRangeHints[PXU_CONTROL9].HintDescriptor
      = (LADSPA_HINT_BOUNDED_BELOW 
	 | LADSPA_HINT_BOUNDED_ABOVE
	 | LADSPA_HINT_DEFAULT_MIDDLE);
    psPortRangeHints[PXU_CONTROL9].LowerBound 
      = 0;
    psPortRangeHints[PXU_CONTROL9].UpperBound
      = 1;

    psPortRangeHints[PXU_CONTROL10].HintDescriptor
      = (LADSPA_HINT_BOUNDED_BELOW 
	 | LADSPA_HINT_BOUNDED_ABOVE
	 | LADSPA_HINT_DEFAULT_MIDDLE);
    psPortRangeHints[PXU_CONTROL10].LowerBound 
      = 0;
    psPortRangeHints[PXU_CONTROL10].UpperBound
      = 1;

    psPortRangeHints[PXU_INPUT1].HintDescriptor
      = 0;
    psPortRangeHints[PXU_OUTPUT1].HintDescriptor
      = 0;
    psPortRangeHints[PXU_INPUT2].HintDescriptor
      = 0;
    psPortRangeHints[PXU_OUTPUT2].HintDescriptor
      = 0;
    g_psStereoDescriptor->instantiate 
      = instantiatePxu;
    g_psStereoDescriptor->connect_port 
      = connectPortToPxu;
    g_psStereoDescriptor->activate
      = NULL;
    g_psStereoDescriptor->run
      = runStereoPxu;
    g_psStereoDescriptor->run_adding
      = NULL;
    g_psStereoDescriptor->set_run_adding_gain
      = NULL;
    g_psStereoDescriptor->deactivate
      = NULL;
    g_psStereoDescriptor->cleanup
      = cleanupPxu;
  }
}

/*****************************************************************************/

void
deleteDescriptor(LADSPA_Descriptor * psDescriptor) {
  unsigned long lIndex;
  if (psDescriptor) {
    free((char *)psDescriptor->Label);
    free((char *)psDescriptor->Name);
    free((char *)psDescriptor->Maker);
    free((char *)psDescriptor->Copyright);
    free((LADSPA_PortDescriptor *)psDescriptor->PortDescriptors);
    for (lIndex = 0; lIndex < psDescriptor->PortCount; lIndex++)
      free((char *)(psDescriptor->PortNames[lIndex]));
    free((char **)psDescriptor->PortNames);
    free((LADSPA_PortRangeHint *)psDescriptor->PortRangeHints);
    free(psDescriptor);
  }
}

/*****************************************************************************/

/* _fini() is called automatically when the library is unloaded. */
void
_fini() {
  deleteDescriptor(g_psMonoDescriptor);
  deleteDescriptor(g_psStereoDescriptor);
}

/*****************************************************************************/

/* Return a descriptor of the requested plugin type. There are two
   plugin types available in this library (mono and stereo). */
const LADSPA_Descriptor * 
ladspa_descriptor(unsigned long Index) {
  /* Return the requested descriptor or null if the index is out of
     range. */
  switch (Index) {
  case 0:
    return g_psMonoDescriptor;
  case 1:
    return g_psStereoDescriptor;
  default:
    return NULL;
  }
}

/*****************************************************************************/

/* EOF */
